// Online Game Player Database Class (c)2004 Niall FitzGibbon
// You may redistribute this file (playerdb.h) or modifications of this file as long as this two-line comment remains intact.
// You may distribute object code resulting from compiling this source code so long as such a credit appears in the documentation for the distribution.

#ifndef PLAYERDB_H
#define PLAYERDB_H

#include <string>
#include <vector>
#include <list>
#include <map>
#include <fstream>

using namespace std;

// in the file, the database is stored with the following structures (with the array of name structs being stored after each
// playerdbfileentry structure)
// in memory, the database is stored as a map with the unique id being the key, and a playerdbentry_t being the value
// the memory version of the structure is not versioned

// when the current version is incremented, the old case statement in the LoadFromFile function switch needs to be changed
// so that it references the version number directly instead of using these macros
// this will allow new versions of the database to load older files, with any new fields defaulting to zero

#define PLAYERDATABASE_SIGNATURE		*(DWORD*)"TPDB"
#define PDB_CURRENT_VERSION				1
#define PDB_CURRENT_HEADER_VERSION		playerdbfileheader_version1_t
#define PDB_CURRENT_ENTRY_VERSION		playerdbfileentry_version1_t
#define PDB_CURRENT_MISC_VERSION		playerdbmisc_version1_t

// this version of the header is read first so that the file version can be determined
// then the read pointer is reset and the correct header version is read
typedef struct playerdbfileheader_basic_s
{
	DWORD dwPlayerDatabaseSignature;
	DWORD dwVersion;
} playerdbfileheader_basic_t;

typedef struct playerdbfileheader_version1_s
{
	DWORD dwPlayerDatabaseSignature;
	DWORD dwVersion;
	int iNumberOfEntries;
} playerdbfileheader_version1_t;

// this structure is read from and written directly to file for each unique player, so it must not contain containers
// or pointers or anything that can't be read or written directly
// it should be versioned, with PDB_CURRENT_MISC_VERSION used for the in-memory version
// for backwards-compatibility, new elements should be added to the end only
typedef struct playerdbmisc_version1_s
{
	DWORD dwFlags;
	char szSteamID[32];
} playerdbmisc_version1_t;

typedef struct playerdbfileentry_version1_s
{
	DWORD dwPlayerUniqueID;
	playerdbmisc_version1_t misc;
	int iNumberOfNames;
} playerdbfileentry_version1_t;

typedef struct playerdbname_s
{
	// number of times the player was logged using this name
	int iCount;
	// name
	string sName;
} playerdbname_t;

// define comparison operators for playerdbname_t so that lists can be sorted
bool operator<(const playerdbname_t &o1, const playerdbname_t &o2);

typedef struct playerdbentry_s
{
	DWORD dwPlayerUniqueID;
	PDB_CURRENT_MISC_VERSION misc;
	list<playerdbname_t> lNames;
} playerdbentry_t;

class CPlayerDatabase
{
public:
	CPlayerDatabase(void);
	~CPlayerDatabase(void);
	void LoadFromFile(char *szFile);
	void SaveToFile(char *szFile);

	// add a name for this unique id
	void AddPlayerName(DWORD dwUniqueID, char *szName);
	// get the most common names for this unique id
	int GetPlayerNames(DWORD dwUniqueID, vector<string> *pvNames, int iNumberOfNames);
	// get the number of unique players in the database
	int GetNumberOfUniquePlayers(void);
	// set the file used to save the database in the destructor
	void SetAutoSaveFile(char *szFile);

private:
	void LoadFromFile_version1(ifstream &f);

	map<DWORD, playerdbentry_t> m_mPlayers;
	char m_szAutoSaveFile[MAX_PATH];
};

extern CPlayerDatabase g_PlayerDatabase;

#endif